#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import os
import sys
import json
import subprocess
from datetime import datetime

print("Content-Type: application/json\n")

# --- Lecture paramètres GET ---
query = os.environ.get("QUERY_STRING", "")
params = dict(p.split("=", 1) for p in query.split("&") if "=" in p)
sid = params.get("sid")
aid = params.get("aid")
prehistory_flag = params.get("prehistory", "").lower() in ("1", "true", "yes")

if not sid:
    print(json.dumps({"error": "missing sid"}))
    sys.exit()

# --- Définir chemins ---
SCRIPT_DIR = os.path.dirname(os.path.abspath(__file__))
ANALYZE_PATH = os.path.join(SCRIPT_DIR, "analyze.py")
SPLINTDATA_PATH = os.path.join(SCRIPT_DIR, "uploads", "splintdata.json")

if not os.path.exists(ANALYZE_PATH):
    print(json.dumps({"error": "analyze.py not found"}))
    sys.exit()

# --- Charger splintdata.json (utf-8-sig pour BOM) ---
splintdata = {}
if os.path.exists(SPLINTDATA_PATH):
    try:
        with open(SPLINTDATA_PATH, "r", encoding="utf-8-sig") as f:
            loaded = json.load(f)
            if isinstance(loaded, dict) and "assets" in loaded:
                splintdata = loaded.get("assets", {})
            else:
                splintdata = loaded
    except Exception as e:
        print(json.dumps({"error": f"Failed to load splintdata.json: {str(e)}"}))
        sys.exit()

# --- Exécuter analyze.py ---
env = os.environ.copy()
env["QUERY_STRING"] = f"sid={sid}"
try:
    result = subprocess.run(["python3", ANALYZE_PATH], env=env, capture_output=True, text=True)
except Exception as e:
    print(json.dumps({"error": f"Failed to run analyze.py: {str(e)}"}))
    sys.exit()

if result.returncode != 0:
    print(json.dumps({"error": "analyze.py failed", "stderr": result.stderr}))
    sys.exit()

raw = result.stdout.strip()
json_start = raw.find("{")
if json_start == -1:
    print(json.dumps({"error": "invalid output from analyze.py", "stdout": raw}))
    sys.exit()

try:
    analyze_data = json.loads(raw[json_start:])
except Exception as e:
    print(json.dumps({"error": f"Invalid JSON from analyze.py: {str(e)}"}))
    sys.exit()

assets = analyze_data.get("assets", {})
transactions = analyze_data.get("transactions", {})

# --- Fonctions utilitaires ---
def convert_period_to_date(period_str):
    try:
        return datetime.strptime(period_str, "%Y%m").strftime("%Y-%m-%01")
    except Exception:
        return None

def parse_date_to_dt(d):
    if not d:
        return None
    d0 = str(d).strip()
    try:
        if len(d0) == 7 and d0.count("-") == 1:
            return datetime.strptime(d0 + "-01", "%Y-%m-%d")
        return datetime.fromisoformat(d0.split("T")[0])
    except Exception:
        pass
    try:
        return datetime.strptime(d0, "%d.%m.%Y")
    except Exception:
        pass
    return None

def dt_to_iso(dt):
    return dt.strftime("%Y-%m-%d") if dt else None

# --- Construction des données graphiques ---
graph_data = {}

for asset_id_key, asset in assets.items():
    if aid and asset_id_key != aid:
        continue

    history = asset.get("history", {})
    txs = transactions.get(asset_id_key, [])
    timeline = []

    # --- Première transaction d'achat ---
    first_buy = next((t for t in txs if "achat" in t.get("type", "").lower() or "buy" in t.get("type", "").lower()), None)

    # --- Dernière transaction de vente ---
    last_sell = next((t for t in reversed(txs) if "vente" in t.get("type", "").lower() or "sell" in t.get("type", "").lower()), None)

    # --- Ajouter la première transaction (achat) ---
    if first_buy:
        date_iso = first_buy.get("date_iso") or first_buy.get("date")
        timeline.append({
            "date": date_iso,
            "price": first_buy.get("price_per_splint"),
            "source": "transaction_buy"
        })

    # --- Ajouter les historiques analyze.py ---
    for period, values in history.items():
        dt = convert_period_to_date(period)
        if dt:
            timeline.append({
                "date": dt,
                "price": values.get("current_splint_value"),
                "source": "analyze"
            })

    # --- Ajouter les données de splintdata ---
    if asset_id_key in splintdata:
        sd_entry = splintdata[asset_id_key]
        sd_tl = sd_entry.get("timeline", []) if isinstance(sd_entry, dict) else sd_entry
        for e in sd_tl:
            d, p = e.get("date"), e.get("price")
            if d and p is not None:
                timeline.append({
                    "date": d,
                    "price": p,
                    "source": e.get("source", "splintdata")
                })

    # --- Ajouter la dernière transaction de vente (si actif vendu) ---
    if (asset.get("status") == "sold" or asset.get("invested", 0) == 0) and last_sell:
        date_iso = last_sell.get("date_iso") or last_sell.get("date")
        timeline.append({
            "date": date_iso,
            "price": last_sell.get("price_per_splint"),
            "source": "transaction_sell"
        })

    # --- Normaliser & trier ---
    normalized = []
    for e in timeline:
        dt = parse_date_to_dt(e.get("date"))
        if not dt or e.get("price") is None:
            continue
        normalized.append({
            "date": dt_to_iso(dt),
            "price": float(e["price"]),
            "source": e.get("source", "unknown")
        })

    normalized.sort(key=lambda x: x["date"])

    # --- Fusion avec priorité transactions ---
    merged = {}
    for e in normalized:
        d = e["date"]
        if d in merged:
            if "transaction" in e["source"]:
                merged[d] = e
        else:
            merged[d] = e

    merged_items = list(merged.values())
    merged_items.sort(key=lambda x: x["date"])

    # --- Exclure avant l'achat (sauf prehistory=1) ---
    if first_buy and not prehistory_flag:
        first_buy_dt = parse_date_to_dt(first_buy.get("date_iso") or first_buy.get("date"))
        merged_items = [m for m in merged_items if not first_buy_dt or parse_date_to_dt(m["date"]) >= first_buy_dt]

    # --- ✅ Si vendu : supprimer toute donnée après la dernière date de vente ---
    if asset.get("status") == "sold" or asset.get("invested", 0) == 0:
        if last_sell:
            last_sell_dt = parse_date_to_dt(last_sell.get("date_iso") or last_sell.get("date"))
            last_sell_iso = dt_to_iso(last_sell_dt)
            # On conserve seulement jusqu'à la date de vente incluse
            merged_items = [m for m in merged_items if parse_date_to_dt(m["date"]) <= last_sell_dt]
            # S'assurer que la vente finale est incluse
            if all(m["date"] != last_sell_iso for m in merged_items):
                merged_items.append({
                    "date": last_sell_iso,
                    "price": last_sell.get("price_per_splint"),
                    "source": "transaction_sell"
                })

    merged_items.sort(key=lambda x: x["date"])

    graph_data[asset_id_key] = {
        "name": asset.get("name"),
        "timeline": merged_items
    }

# --- Sortie JSON finale ---
print(json.dumps({
    "session_id": sid,
    "assets": graph_data
}, ensure_ascii=False, indent=2))
