#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import os, csv, json, cgi, re, traceback
from collections import defaultdict, deque
from datetime import datetime

print("Content-Type: application/json\n")

try:
    BASE_DIR = os.path.dirname(os.path.abspath(__file__))
    UPLOADS_DIR = os.path.join(BASE_DIR, "uploads")

    # --- Récupération du SID ---
    form = cgi.FieldStorage()
    sid = form.getfirst("sid")
    if not sid:
        print(json.dumps({"error": "Identifiant de session (sid) manquant"}))
        raise SystemExit

    session_path = os.path.join(UPLOADS_DIR, sid)
    if not os.path.isdir(session_path):
        print(json.dumps({"error": f"Dossier de session introuvable ({sid})"}))
        raise SystemExit

    # --- Utilitaires ---
    def lire_csv(path):
        with open(path, encoding="utf-8-sig") as f:
            return list(csv.DictReader(f))

    def to_float(v):
        if v is None:
            return 0.0
        v = str(v)
        clean = re.sub(r"[^0-9,.\-]", "", v).replace(",", ".")
        try:
            return float(clean) if clean else 0.0
        except:
            return 0.0

    def extract_month_from_filename(fn):
        m = re.match(r"(\d{6})", fn)
        return m.group(1) if m else None

    def parse_date(s):
        """Try several date formats, return (yyyy,mm,dd) tuple for sorting."""
        if not s:
            return (0,0,0)
        s = s.strip()
        # Common patterns: YYYY-MM-DD..., DD.MM.YYYY, DD/MM/YYYY, YYYY/MM/DD
        try:
            # ISO-like first
            dt = datetime.fromisoformat(s.split("+")[0])
            return (dt.year, dt.month, dt.day)
        except:
            pass
        # dd.mm.yyyy
        m = re.match(r'(\d{1,2})\.(\d{1,2})\.(\d{4})', s)
        if m:
            d,mn,y = m.groups()
            return (int(y), int(mn), int(d))
        # dd/mm/yyyy
        m = re.match(r'(\d{1,2})/(\d{1,2})/(\d{4})', s)
        if m:
            d,mn,y = m.groups()
            return (int(y), int(mn), int(d))
        # fallback
        return (0,0,0)

    def normalize_date_to_iso(s):
        """
        Retourne YYYY-MM-DD si possible, sinon None.
        Accepte formats: YYYY-MM-DD, YYYYMM, YYYYMMDD, DD.MM.YYYY, DD/MM/YYYY, ISO with time.
        """
        if not s:
            return None
        s0 = str(s).strip()
        # ISO-like
        try:
            dt = datetime.fromisoformat(s0.split("+")[0])
            return dt.strftime("%Y-%m-%d")
        except:
            pass
        # YYYYMMDD
        m = re.match(r'^(\d{4})(\d{2})(\d{2})$', s0)
        if m:
            y,mo,da = m.groups()
            return f"{y}-{mo}-{da}"
        # YYYYMM
        m = re.match(r'^(\d{4})(\d{2})$', s0)
        if m:
            y,mo = m.groups()
            return f"{y}-{mo}-01"
        # dd.mm.yyyy
        m = re.match(r'^(\d{1,2})\.(\d{1,2})\.(\d{4})$', s0)
        if m:
            d,mo,y = m.groups()
            return f"{y}-{int(mo):02d}-{int(d):02d}"
        # dd/mm/yyyy
        m = re.match(r'^(\d{1,2})/(\d{1,2})/(\d{4})$', s0)
        if m:
            d,mo,y = m.groups()
            return f"{y}-{int(mo):02d}-{int(d):02d}"
        return None

    # --- Découverte des fichiers ---
    activities_files = []
    opportunities_files = {}  # month -> path

    for fname in os.listdir(session_path):
        path = os.path.join(session_path, fname)
        lf = fname.lower()
        if "activit" in lf and fname.endswith(".csv"):
            activities_files.append(path)
        elif ("opportunit" in lf or "investment" in lf) and fname.endswith(".csv"):
            mo = extract_month_from_filename(fname)
            if mo:
                opportunities_files[mo] = path

    if not activities_files:
        print(json.dumps({"error":"Aucun fichier user_activities trouvé"}))
        raise SystemExit

    # --- Lire activities (concat de tous) ---
    activities = []
    for p in activities_files:
        activities.extend(lire_csv(p))

    # --- Dé-duplication simple (Transaction ID) pour éviter double comptage si upload multiples ---
    seen_tx = set()
    uniq_acts = []
    for row in activities:
        txid = (row.get("Transaction ID") or "").strip()
        if txid:
            key = f"tx::{txid}"
        else:
            # fallback: date|asset|amount|price
            key = "row::" + "|".join([
                (row.get("Date Of Transaction") or "").strip(),
                (row.get("Asset ID") or "").strip(),
                str(row.get("Money Amount") or ""),
                str(row.get("Purchase/Sale Price Per Splint") or "")
            ])
        if key in seen_tx:
            continue
        seen_tx.add(key)
        uniq_acts.append(row)
    activities = uniq_acts

    # --- Trier les activities par date croissante (chronologie) ---
    activities.sort(key=lambda r: parse_date(r.get("Date Of Transaction") or r.get("Date") or ""))

    # --- Lire opportunities par mois (history) ---
    opportunities_by_month = {}
    for mo, p in opportunities_files.items():
        opportunities_by_month[mo] = lire_csv(p)
    months = sorted(opportunities_by_month.keys())
    latest_month = months[-1] if months else None

    # --- Reconstituer positions via FIFO lots à partir des activities ---
    # Structure: portfolio[aid]["lots"] = deque of (qty, unit_cost)
    portfolio = defaultdict(lambda: {"lots": deque(), "realized_proceeds": 0.0, "realized_profit": 0.0})

    for row in activities:
        aid = (row.get("Asset ID") or "").strip()
        if not aid:
            continue
        ttype = (row.get("Transaction Type") or "").lower()
        amount = to_float(row.get("Money Amount"))
        unit_price = to_float(row.get("Purchase/Sale Price Per Splint"))

        # if price missing, skip
        if unit_price == 0:
            continue
        qty = round(amount / unit_price, 8)

        if "achat" in ttype or "buy" in ttype:
            # append lot (qty, unit_cost)
            portfolio[aid]["lots"].append((qty, unit_price))
        elif "vente" in ttype or "sell" in ttype:
            # sale: consume FIFO
            remains = qty
            proceeds = amount
            portfolio[aid]["realized_proceeds"] += proceeds
            profit = 0.0
            while remains > 0 and portfolio[aid]["lots"]:
                lot_qty, lot_price = portfolio[aid]["lots"][0]
                if lot_qty > remains + 1e-9:
                    # partial consume
                    profit += (unit_price - lot_price) * remains
                    portfolio[aid]["lots"][0] = (round(lot_qty - remains, 8), lot_price)
                    remains = 0
                else:
                    # consume entire lot
                    profit += (unit_price - lot_price) * lot_qty
                    portfolio[aid]["lots"].popleft()
                    remains = round(remains - lot_qty, 8)
            # if remains > 0 and no lots -> sold more than bought (we ignore negative lots)
            portfolio[aid]["realized_profit"] += profit

    # --- Construire asset_data à partir des opportunities history (valeurs par mois) ---
    asset_data = {}
    # Initialize from all months so sold assets are included in history
    for mo, rows in opportunities_by_month.items():
        for opp in rows:
            aid = (opp.get("Asset ID") or "").strip()
            if not aid:
                continue
            owned = to_float(opp.get("Owned Splints"))
            cur_splint_val = to_float(opp.get("Current Splint Value"))
            avg_buy = to_float(opp.get("Average Purchase Price Per Splint"))
            name = opp.get("Asset Name") or aid
            cur_total = round(owned * cur_splint_val, 2)
            if aid not in asset_data:
                asset_data[aid] = {
                    "name": name,
                    "history": {},
                    "owned_splints": 0.0,
                    "invested": 0.0,
                    "realized_proceeds": 0.0,
                    "gain_realise": 0.0,
                    "gain_latent": 0.0,
                    "average_buy_price": avg_buy,
                    "current_splint_value": 0.0,
                    "current_value": 0.0,
                    "performance": 0.0,
                    "status": "unknown"
                }
            asset_data[aid]["history"][mo] = {
                "owned_splints": owned,
                "current_splint_value": cur_splint_val,
                "average_buy_price": avg_buy,
                "current_value": cur_total
            }

    # Ensure assets present in activities but not in opportunities are also represented
    for aid in portfolio.keys():
        if aid not in asset_data:
            asset_data[aid] = {
                "name": aid,
                "history": {},
                "owned_splints": 0.0,
                "invested": 0.0,
                "realized_proceeds": 0.0,
                "gain_realise": 0.0,
                "gain_latent": 0.0,
                "average_buy_price": 0.0,
                "current_splint_value": 0.0,
                "current_value": 0.0,
                "performance": 0.0,
                "status": "unknown"
            }

    # --- Final : compute cost_basis of remaining lots, realized profits/proceeds, latent gains ---
    total_invested = total_realized_profit = total_realized_proceeds = total_current = total_gain_latent = 0.0

    for aid, entry in asset_data.items():
        # cost basis of remaining lots from portfolio
        lots = portfolio.get(aid, {"lots": deque(), "realized_proceeds": 0.0, "realized_profit": 0.0})["lots"]
        cost_basis = 0.0
        remaining_qty = 0.0
        for q, p in lots:
            cost_basis += q * p
            remaining_qty += q

        # realized
        realized_proceeds = portfolio.get(aid, {}).get("realized_proceeds", 0.0)
        realized_profit = portfolio.get(aid, {}).get("realized_profit", 0.0)

        # value now from latest month if available, else last known month
        if latest_month and latest_month in entry["history"]:
            hist = entry["history"][latest_month]
        elif entry["history"]:
            last_m = sorted(entry["history"].keys())[-1]
            hist = entry["history"][last_m]
        else:
            hist = {"owned_splints": 0.0, "current_splint_value": 0.0, "average_buy_price": 0.0, "current_value": 0.0}

        current_splint_val = hist.get("current_splint_value", 0.0)
        current_value = hist.get("current_value", 0.0)

        # If activities show remaining_qty (from FIFO reconstruction), use it; otherwise fall back to historical owned
        owned_splints = remaining_qty if remaining_qty is not None else hist.get("owned_splints", 0.0)

        # If last-month opportunities show owned_splints different from computed, we trust activities (chronology)
        # But for valuation we use the opportunities price (current_splint_val)
        current_value = round(owned_splints * current_splint_val, 2)

        gain_latent = current_value - cost_basis
        performance = round(gain_latent + realized_profit, 2)

        # update entry
        entry["name"] = entry.get("name") or aid
        entry["owned_splints"] = round(owned_splints, 8)
        entry["invested"] = round(cost_basis, 2)
        entry["realized_proceeds"] = round(realized_proceeds, 2)
        entry["gain_realise"] = round(realized_profit, 2)
        entry["gain_latent"] = round(gain_latent, 2)
        entry["average_buy_price"] = round(entry.get("average_buy_price", 0.0), 2)
        entry["current_splint_value"] = round(current_splint_val, 2)
        entry["current_value"] = round(current_value, 2)
        entry["performance"] = performance
        entry["status"] = "owned" if owned_splints > 0 else "sold"

        total_invested += cost_basis
        total_realized_profit += realized_profit
        total_realized_proceeds += realized_proceeds
        total_current += current_value
        total_gain_latent += gain_latent

    global_result = {
        "invested": round(total_invested, 2),
        "realized_proceeds": round(total_realized_proceeds, 2),
        "gain_realise": round(total_realized_profit, 2),
        "gain_latent": round(total_gain_latent, 2),
        "total_performance": round(total_realized_profit + total_gain_latent, 2)
    }

    # =====================================================
    # Ajout NON-invasif : transactions détaillées par actif (avec date normalisée)
    # =====================================================
    transactions_by_asset = {}
    for row in activities:
        aid = (row.get("Asset ID") or "").strip()
        if not aid:
            continue

        t_id = (row.get("Transaction ID") or "").strip()
        t_type = (row.get("Transaction Type") or "").strip()
        date = (row.get("Date Of Transaction") or row.get("Date") or "").strip()
        date_iso = normalize_date_to_iso(date)  # <= ajouté : date normalisée pour le front
        amount = to_float(row.get("Money Amount"))
        price_per_splint = to_float(row.get("Purchase/Sale Price Per Splint"))
        fees = to_float(row.get("Fees") or row.get("Fees (EUR)") or row.get("Fees (EUR)") or 0)

        tx = {
            "id": t_id,
            "type": t_type,
            "date": date,
            "date_iso": date_iso,
            "amount": round(amount, 2),
            "price_per_splint": round(price_per_splint, 2),
            "fees": round(fees, 2)
        }

        transactions_by_asset.setdefault(aid, []).append(tx)

    result = {
        "sid": sid,
        "months": months,
        "latest_month": latest_month,
        "global": global_result,
        "assets": asset_data,
        "transactions": transactions_by_asset
    }

    print(json.dumps(result, ensure_ascii=False, indent=2))

except Exception as e:
    print(json.dumps({
        "error": str(e),
        "trace": traceback.format_exc().splitlines()[-10:]
    }, ensure_ascii=False))
